% analyzeBehaviorVideo.m

% This function takes a grayscale behavioral video as input and produces
% several graphs showing how the annotations vary over time.  We use this
% script to analyze a video of a worm's pharyngeal response to light.  This
% program is flexible, because it allows you to specify which events you
% want to annotate when you run the program.  For example, let's say that
% you want to record the times of one of the following events in a video 
% (though in fact you can record any arbitrary event you want):
% - pump
%   - pump onset (first frame of posterior movement of grinder)
%   - pump contraction (frame of most posterior position of grinder)
%   - pump relaxation (last frame in which the grinder moves during relaxation)
% - gargle (a spit that occurs due to the pharynx sucking in and spitting out mineral oil
%   - gargle onset
%   - gargle release
% - spit (either a gargle or true spit of outward movement of fluid that was previously in the pharynx
% - inward particle (mineral oil) movement
% - neutral particle movement
% - outward particle movement
% - isthmus peristalsis
% - opening of the buccal valve (aka "jaws")
% - closing of the buccal valve
% - partial regurgitation (aka "gastric ping-pong")
% - and more... (such as true vomiting)

% To record a few events such as pumping and spitting, you would use the 
% program as so.  First, call the program:
%   > analyzeBehaviorVideo
%   > Video to analyze: behaviorVideo.avi
%   > Mark specified events with a specific key press: {'pump' 'P' 'spit' 'S'}
%   > You can mark the start and end of unscorable sections with the [ and ] keys.
%   > You can unmark an event by pressing the same key again in the same frame.

% Next, a video frame will appear with a column on the right.  It will say
% whether the current frame in the video has a mark specified for it or
% not.  

% After marking all frames in the video, you can close the video
% viewer.  Graphs will be generated showing the moving average of the data
% collected for each mark, and a data file will be saved with the
% timestamps of all marked events (just like real-time pumping analysis gives).

% PARAMETERS
winWidth = 1000;  % Plot moving average over this time window
numMsInS = 1000;
yMin = 0;
yMax = 7;
missColor = [200 200 200] ./ 255;
startDelay = 10;
stimLen = 10;
yBar = 6.5;
lightc = [98 0 255] ./ 255;  %VIOLET % [98 0 255] ./ 255; %BLUE          % Color of light bar
tic;

disp('   INSTRUCTIONS:');
disp('   This program will help you quantitatively analyze events in a video.');
disp('   You will annotate events using specific keys on the keyboard.');
disp('   You can mark frames in which the subject is missing with the M key.');
disp('   You can also mark the start and end of unscorable sections with the [ and ] keys.');
disp('   You can unmark any event by pressing the same key again in the same frame.');
videoFileName = input('Video to analyze: ', 's');
saveFileName = input('Save analysis using this filename (leave blank): ', 's');
eventkeys = input('Event names and keys (e.g. {''Pump'' ''P''}: ');

% Check to make sure eventkeys are properly specified
if (mod(length(eventkeys), 2) == 1)
    error('The number of entries for the event names and keys must be a multiple of 2.');
end
if (length(eventkeys) < 2)
    error('The number of entries for the event names and keys must be at least 2.');
end
for i=2:2:length(eventkeys)
    if (length(eventkeys{i}) > 1)
        error('The key associated with an event must be exactly one character long.');
    end
    if (~isempty(strfind(eventkeys{i}, '[')) || ~isempty(strfind(eventkeys{i}, ']'))...
            || ~isempty(strfind(eventkeys{i}, 'm')))
        error('The bracket keys [ and ] and m are reserved for marking frames in which the worm is missing.');
    end
end

if (isempty(saveFileName))
    saveFileName = videoFileName;
end

[p vidName ext] = fileparts(saveFileName);

vreader = VideoReader(videoFileName);
fr = 1000; %vreader.FrameRate;
numFrames = vreader.NumberOfFrames;
vidWidth = vreader.Width;
vidHeight = vreader.Height;

numEventTypes = length(eventkeys)/2;

% Start up the GUI
h = layout_behavior(vreader, eventkeys);
waitfor(h, 'Pointer', 'watch');

handles = guidata(h);

% handles.events contains a 1 in the frame in which the corresponding event occurred
% convert events to timestamps, then plot
data = cell(numEventTypes, 1);
figs = zeros(numEventTypes, 1);

% Process missing data to extract shadable regions
d = diff(handles.events(:,end));
missLeft = round(find(d == 1) ./ fr .* numMsInS);
missRight = round(find(d == -1) ./ fr .* numMsInS);
if ((isempty(missLeft) && ~isempty(missRight)) || ...
        (~isempty(missLeft) && ~isempty(missRight) && missLeft(1) > missRight(1)))
    missLeft = [1; missLeft];
end
if ((isempty(missRight) && ~isempty(missLeft)) || ...
        (~isempty(missRight) && ~isempty(missLeft) && missRight(end) < missLeft(end)))
    missRight(end+1) = round(numFrames ./ fr .* numMsInS);
end

for i=1:numEventTypes
    data{i} = round(find(handles.events(:,i)) ./ fr .* numMsInS);  % convert data to ms
    [m sd se figs(i)] = eventsOverTime(data(i), round(numFrames / fr * numMsInS), winWidth, numMsInS, ...
                               eventkeys{2*i-1}, true);
    for j = 1:length(missLeft)
        p = patch([missLeft(j) missRight(j) missRight(j) missLeft(j)] ./ numMsInS, [yMin yMin yMax yMax]', ...
            missColor, 'LineStyle', 'none');
    end
    ylim([yMin yMax]);
    hold on;
	plot([startDelay startDelay+stimLen], [yBar yBar], 'LineWidth', 16, 'Color', lightc);
    text(startDelay+stimLen/2, yBar, 'light',...
        'FontWeight', 'bold', 'FontSize', 10, 'Color', [1 1 1], ...
        'HorizontalAlignment', 'center');
end


eval([vidName ' = data;']);
if (exist([vidName '.mat'], 'file') == 0)  % Don't overwrite if there's an existing file
    save([vidName, '.mat'], vidName, 'eventkeys', 'missLeft', 'missRight');
else
    disp('Did not save data!  File already exists.');
    beep;
end

for i=1:length(figs)
	saveas(figs(i), [vidName '_' eventkeys{2*i-1} 's.fig']);
end

t = toc;
disp([num2str(round(t/60)) ' minutes taken.']);
beep;
